<?php
/* --------------------------------------------------------------
  config.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

/**
 * Class StyleEditConfig
 *
 * Contains basic configuration for StyleEdit 3.
 */
class StyleEditConfig
{
	/**
	 * Relative Path To StyleEdit Directory
	 *
	 * This path will be used for referencing StyleEdit3 resources in a relative manner directly within
	 * the CSS/SCSS (e.g. images). Provide the relative path that will finally resolve to the StyleEdit
	 * directory (without a trailing slash).
	 *
	 * @var string
	 */
	const RELATIVE_PATH = '../../StyleEdit3/';
	
	/**
	 * Default App Language
	 *
	 * Can be either 'de' and 'en' and it is used if there is no GET
	 * parameter specifying another language.
	 *
	 * Default value: 'de'
	 *
	 * @var string
	 */
	const DEFAULT_LANGUAGE = 'de';
	
	/**
	 * Authorization Session Variable Name
	 *
	 * The user that is going to use StyleEdit must have this key in his
	 * session otherwise he is not authorized to use the application.
	 *
	 * Default value: 'STYLE_EDIT_AUTH'
	 *
	 * @var string
	 */
	const AUTH_SESSION_KEY = 'STYLE_EDIT_AUTH';
	
	/**
	 * Authorization Session Variable Value
	 *
	 * The user that is going to use StyleEdit must have this value in his
	 * session authorization variable otherwise he is not authorized to use the application.
	 *
	 * Default value: true
	 *
	 * @var mixed
	 */
	const AUTH_SESSION_VALUE = true;
	
	/**
	 * API Response Type in Error Case
	 *
	 * JSON response on false, HTML response on true
	 *
	 * Default value: false
	 *
	 * @var bool
	 */
	const DEBUG_HTML = false;
	
	/**
	 * API Error Backtrace
	 *
	 * If StyleEditConfig::DEBUG_HTML is false a complete error backtrace can be added in the JSON response.
	 *
	 * Default value: false
	 *
	 * @var bool
	 */
	const DEBUG_BACKTRACE = false;
	
	/**
	 * Relative path to the images directory with trailing slash beginning from the templates directory
	 *
	 * Default value: 'assets/images/custom/'
	 *
	 * @var string
	 */
	const IMAGES_DIR = 'assets/images/custom/';
	
	/**
	 * Relative path to the images directory with trailing slash beginning from the themes directory
	 *
	 * Default value: 'images/custom/'
	 *
	 * @var string
	 */
	const THEME_IMAGES_DIR = 'images/custom/';
	
	
	/**
	 * Returns Session Save Path
	 *
	 * Directory where the session cookie is stored.
	 *
	 * Default value: realpath(dirname(__DIR__) . '/cache') . '/'
	 *
	 * @return string
	 */
	public static function getSessionSavePath()
	{
		return realpath(dirname(__DIR__) . '/cache') . '/';
	}
	
	
	/**
	 * Returns cache directory path.
	 *
	 * Default value: realpath(dirname(__DIR__) . '/cache') . '/'
	 *
	 * @return bool|string
	 */
	public static function getCacheDirectoryPath()
	{
		return realpath(dirname(__DIR__) . '/cache') . '/';
	}
	
	
	/**
	 * Returns CSS Cache File Path
	 *
	 * File path of the compiled css file.
	 *
	 * Default value: realpath(dirname(__DIR__) . '/cache/') . '__dynamics.css';
	 *
	 * @return string
	 */
	public static function getCssCacheFilePath()
	{
		return realpath(dirname(__DIR__) . '/cache') . '/__dynamics.css';
	}
	
	
	/**
	 * Returns Images Directory
	 *
	 * Relative path to the images directory with trailing slash
	 *
	 * Default value: \StyleEdit\StyleThemeControl::getImagesDirectory($p_templateName)
	 *
	 * @param string $p_templateName
	 *
	 * @return string
	 */
	public static function getImagesDirectory($p_templateName)
	{
		return \StyleEdit\StyleThemeControl::getImagesDirectory($p_templateName);
	}
	
	
	/**
	 * Returns Images Directory Path
	 *
	 * Absolute file path to the image directory without trailing slash
	 *
	 * Default value: realpath(__DIR__ . \StyleEdit\StyleThemeControl::getImagesDirectoryPath($p_templateName))
	 *
	 * @param string $p_templateName
	 *
	 * @return string
	 */
	public static function getImagesDirectoryPath($p_templateName)
	{
		return realpath(__DIR__ . \StyleEdit\StyleThemeControl::getImagesDirectoryPath($p_templateName));
	}
	
	
	/**
	 * Returns the base path for additional boxes
	 *
	 * Absolute file path to the base path to a directory where additional boxes are located
	 *
	 * Default value: realpath(dirname(__DIR__) . '/GXModules/')
	 *
	 * @return string
	 */
	public static function getAdditionalBoxesBasePath()
	{
		return realpath(dirname(__DIR__) . '/GXModules') . '/';
	}
	
	
	/**
	 * Returns path for media directory
	 *
	 * Absolute path to the directory where the secure token file resides
	 *
	 * @return string
	 */
	public static function getMediaPath()
	{
		return dirname(__DIR__) . '/media/';
	}
	
	
	/**
	 * Returns the directory for the additional boxes
	 *
	 * Relative file path to the directory of additional boxes
	 *
	 * @param string $p_templateName
	 *
	 * @return string
	 */
	public static function getAdditionalBoxesDirectoryPath($p_templateName = null)
	{
		if($p_templateName !== null)
		{
			return \StyleEdit\StyleThemeControl::isThemeSystemActive() ? '/Themes/' . $p_templateName . '/Boxes/' : '/Templates/' . $p_templateName . '/Boxes/';
		}
		
		return \StyleEdit\StyleThemeControl::isThemeSystemActive() ? '/Themes/All/Boxes/' : '/Templates/All/Boxes/';
	}
}
